// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_MESSAGE_ROUTER_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_MESSAGE_ROUTER_H

#include "common.h"
#include "SslWrapper.h"
#include "ChannelManager.h"

class ProtocolEndpointBase;

/**
 * @internal
 * The MessageRouter is exactly what it sounds like, it maintains the channel:service_id
 * mappings and figures out which ProtocolEndpoint should get the message that was just
 * received.
 * It is worth noting that the word ProtocolEndpoint is used to represent a 'Service' in
 * the protocol specification. There are two reasons for this - 1) The word service means
 * something specific in a lot of contexts, so we avoid it. 2) The actual service names
 * are used up by the protocol buffer objects, so it is just simpler this way.
 */
class MessageRouter {
public:
    bool init(ChannelManager* channelManager);
    void shutdown();
    int queueOutgoing(uint8_t channelId, void* buf, size_t len);
    int queueOutgoingUnencrypted(uint8_t channelId, void* buf, size_t len);
    int routeMessage(uint8_t channelId, const shared_ptr<IoBuffer>& msg);
    int routeChannelControlMsg(const shared_ptr<Frame>& frame, void* message, size_t len);
    bool registerService(ProtocolEndpointBase* service);
    void populateServiceDiscoveryResponse(ServiceDiscoveryResponse* sdr);
    void marshallProto(uint16_t type, const google::protobuf::MessageLite& proto, IoBuffer* out);
    void setupMapping(uint8_t serviceId, uint8_t channelId);
    void unrecoverableError(MessageStatus err);
    inline void setSslWrapper(SslWrapper* sslWrapper) {
        mChannelManager->setSslWrapper(sslWrapper);
    }
    bool closeChannel(uint8_t channelId);

private:
    int handleChannelOpenReq(uint8_t channelId, const ChannelOpenRequest& req);
    int sendChannelOpenResp(uint8_t channelId, int32_t status);
    int handleChannelCloseNotif(uint8_t channelId, const ChannelCloseNotification& notification);
    int sendUnexpectedMessage(uint8_t channelId);
    uint16_t extractType(uint8_t* message);
    void notifyChannelOpened(uint8_t channelId, const ChannelOpenRequest& req);

    uint8_t mChannelServiceIdMap[MAX_CHANNELS];
    ProtocolEndpointBase* mServiceMap[MAX_SERVICES];
    ChannelManager* mChannelManager;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_MESSAGE_ROUTER_H
